//**********************************************************************
// file			: MQ6801_Flash.c
// version		: V1.1 2021/02/24
// brief		: flash memory related functions
// note			: make sure that the chip does not power off or reset before the flash memory related functions are completely executed
//**********************************************************************

#include "main.h"

//==============================================================================
// name			: Flash_WriteByte 
// retval		: 0:write success  1:write fail<difference from expected value after writing>  2:write fail<not 0xFF before writing>
//            3:invalid address parameter
// param		: <addr>	specified address of flash memory(0x7E80~0x7EFF)
//     		  <data>	one byte of data to be written to flash memory
// note			: make sure the address to be written has no code, so as not to modify the structure of the program after writing
//==============================================================================
unsigned char Flash_WriteByte(unsigned int addr,unsigned char data)
{	
	unsigned char tmpIMF;
	
	
	if( !((addr>=0x7e80&&addr<=0x7eff) ||
		   (addr>=0x8000&&addr<=0xffff)) )
		return 3;				// invalid address
		   
	if( *((unsigned char*)addr) != 0xff)
		return 2;				// not 0xFF before writing
		   
    FLSCR1_FLSMD = 0x05;        // enable command sequence and toggle execution
    FLSCR2       = 0xd5;        // enable a change in the FLSCR1 setting
	tmpIMF = EI_IMF;
	EI_IMF = 0x0;			//DI
	__asm("ld (0xf555),0xaa");
	__asm("ld (0xfaaa),0x55");
	__asm("ld (0xf555),0xa0");
	*(unsigned char* )(addr) = data;      
	EI_IMF = tmpIMF;		//recover IMF
    FLSCR1_FLSMD = 0x02;        // disable command sequence and toggle execution
    FLSCR2       = 0xd5;        // enable a change in the FLSCR1 setting
	
	if( *((unsigned char*)addr) != data)
		return 1;				// write fail
	
	
	return 0;					// write success
	
}


//==============================================================================
// name			: Flash_ReadSequence
// retval		: void
// param		: <addr> 	specified start address of flash memory
//     		  <data> 	the values of specified addresses of flash memory
//     		  <length>	read data length
// note			: 
//==============================================================================
void Flash_ReadSequence(unsigned int addr,unsigned char* data,unsigned int length)
{
	unsigned char *p_addr = (unsigned char *)addr;
	unsigned int i;
	for(i=0;i<length;i++)
	{
		data[i] = *(p_addr+i);
		WDCDR=0x4e; 			//clear wdt
	}
	
}




//==============================================================================
// name			: FlashReadByte 
// retval		: unsigned char
// param		: <addr>	specified address of flash memory
// note			: the value of specified address of flash memory
//==============================================================================
unsigned char Flash_ReadByte(unsigned int addr)
{	
	return *((unsigned char *)addr);
}




//==============================================================================
// name			: Flash_SectionErase 
// retval		: 0:erase success  1:erase fail 3:invalid address parameter
// param		: <addr> specified address of flash memory area to be erased
// note			: The flash memory area to be erased is specified by the upper 9 bits of address(128 Bytes)
//     
//     		  ex. if address =0x7e80 means section:0x7e80~0x7eff including 0x7e80, so erase area is 0x7e80~0x7eff  (128 Bytes)
//         		  if address =0x7e99 means section:0x7e80~0x7eff including 0x7e99, so erase area is 0x7e80~0x7eff  (128 Bytes)
//         		  if address =0x7eff means section:0x7e80~0x7eff including 0x7eff, so erase area is 0x7e80~0x7eff  (128 Bytes) 
//         		  if address =0xf000 means section:0xf000~0xf07f including 0xf000, so erase area is 0xf000~0xf07f  (128 Bytes)
//         		  if address =0xcfff means section:0xcf80~0xcfff including 0xcfff, so erase area is 0xcf80~0xcfff  (128 Bytes)
// 
//         	  section rule:0xXX00~0xXX7f , 0xXX80~0xXXff
// 
// 		   	  data flash memory area:
//         		  <section>0x7e80~0x7eff
//  	      program flash memory area:
//         		  <section>0x8000~0x807f 
//         		  <section>0x8080~0x80ff 
//         		  <section>0x8100~0x817f 
//         		  <section>0x8180~0x81ff
//                          :
//                          :						  
//         		  <section>0xff00~0xff7f 
//         		  <section>0xff80~0xffff
//==============================================================================
unsigned char Flash_SectionErase(unsigned int addr)
{
	unsigned char tmpIMF;
	unsigned int  iAddr,startAddr,endAddr;
	
	
	if( !((addr>=0x7e80&&addr<=0x7eff) ||
		   (addr>=0x8000&&addr<=0xffff)) )
		return 3;				// invalid address
	
    FLSCR1_FLSMD = 0x05;        // enable command sequence and toggle execution
    FLSCR2       = 0xd5;        // enable a change in the FLSCR1 setting
	tmpIMF = EI_IMF;
	EI_IMF = 0x0;			//DI
	__asm("ld (0xf555),0xaa");
	__asm("ld (0xfaaa),0x55");
	__asm("ld (0xf555),0x80");
	__asm("ld (0xf555),0xaa");
	__asm("ld (0xfaaa),0x55");		
	*(unsigned char* )(addr) = 0x20;
	EI_IMF = tmpIMF;		//recover IMF
    FLSCR1_FLSMD = 0x02;        // disable command sequence and toggle execution
    FLSCR2       = 0xd5;        // enable a change in the FLSCR1 setting
	
	startAddr 	= addr & 0xff80;
	endAddr		= addr|0x7f	;	
	for(iAddr=startAddr;iAddr <=endAddr ;iAddr++){
		if( *(unsigned char* )(iAddr) != 0xff){
			return 1;			// erase fail
		}
	}
	
	return 0;
}








